*** 
*** Code for "The Labor Market Integration of Refugee Migrants in High-Income Countries"
*** Courtney Brell, Christian Dustmann, and Ian Preston
***
*** Analysis of the Building a New Life in Australia survey
*** This file should be run in the folder containing the BNLA data
***

********************************************************************************
*** Preliminaries

clear all

* Import all waves of the survey
use "3.Stata - BNLA Release 4.0\Stata\bnla_a40gr.dta", clear
rename a* *
gen wave=1
save "temp.dta", replace
use "3.Stata - BNLA Release 4.0\Stata\bnla_b40gr.dta", clear
rename b* *
gen wave=2
append using "temp.dta"
save "temp.dta", replace
use "3.Stata - BNLA Release 4.0\Stata\bnla_c40gr.dta", clear
rename c* *
gen wave=3
append using "temp.dta"
save "temp.dta", replace
use "3.Stata - BNLA Release 4.0\Stata\bnla_d40gr.dta", clear
rename d* *
gen wave=4
append using "temp.dta"

* Generate demographic variables
gen countryoforigin=zcob_sacc
label values countryoforigin zcob_sacc
gen female=.
replace female=1 if gender==2
replace female=0 if gender==1

* Calculate the number of years since arrival
recode time_since_arrival (1 2 3 = 0) (4 = 1) (5 = 2) (6 = 3) (7 = 4) (8 = 5) (else = .), gen(yearssincearrive)

* All migrants in this sample are refugees
gen migranttype=1
label define Lmigrant 0 "Native" 1 "Refugee" 2 "Other immigrant"
label values migranttype Lmigrant

* Choose our sample
keep if age>=20&age<=64
* Keep only respondents, others have no LM data
keep if resp==1
* Use survey weights
gen indweight=wgt

********************************************************************************
*** Calculate labor market outcomes

* Employment
gen employment=.
replace employment=0 if em01==2
replace employment=1 if em01==1

* Wages (reported in weekly wage)
gen wage=em_income if em_income>0
replace wage=. if employment==0
gen yearofinterview=2012+wave

*** Convert any net wages to gross
/* 
Relevant marginal tax rates:
2013/14-2015/16:
<=18200:0%
<=37000:19%
<=80000:32.5%
<=180000:37%
>180000:45% (+2% Temporary Budget Repair Levy in 2014/15 & 2015/16)
2016/17-2017/18:
<=18200:0
<=37000:19%
<=87000:32.5%
<=180000:37%
>180000:45% (+2% Temporary Budget Repair Levy in 2016/17)
Converting the cutoffs from gross to net wages:
2013/14-2015/16:
18200;33428;62453;125453
2016/17-2017/18:
18200;33428;67178;125768
*/
gen annualwage=wage*365.2425/7
gen pretaxannualwage=.
replace pretaxannualwage = min(18200,annualwage)+(1/(1-0.19))*max(0,min(33428-18200,annualwage-18200))+(1/(1-0.325))*max(0,min(62453-33428,annualwage-33428))+(1/(1-0.37))*max(0,min(125453-62453,annualwage-62453))+(1/(1-0.45))*max(0,annualwage-125453) if !missing(annualwage)&yearofinterview==2013
replace pretaxannualwage = min(18200,annualwage)+(1/(1-0.19))*max(0,min(33428-18200,annualwage-18200))+(1/(1-0.325))*max(0,min(62453-33428,annualwage-33428))+(1/(1-0.37))*max(0,min(125453-62453,annualwage-62453))+(1/(1-0.45-0.02))*max(0,annualwage-125453) if !missing(annualwage)&(yearofinterview==2014|yearofinterview==2015)
replace pretaxannualwage = min(18200,annualwage)+(1/(1-0.19))*max(0,min(33428-18200,annualwage-18200))+(1/(1-0.325))*max(0,min(67178-33428,annualwage-33428))+(1/(1-0.37))*max(0,min(125453-67178,annualwage-67178))+(1/(1-0.45-0.02))*max(0,annualwage-125453) if !missing(annualwage)&yearofinterview==2016
replace pretaxannualwage = min(18200,annualwage)+(1/(1-0.19))*max(0,min(33428-18200,annualwage-18200))+(1/(1-0.325))*max(0,min(67178-33428,annualwage-33428))+(1/(1-0.37))*max(0,min(125768-67178,annualwage-67178))+(1/(1-0.45))*max(0,annualwage-125768) if !missing(annualwage)&yearofinterview==2017

gen grosswage=wage if em08d==1|em11d==1
replace grosswage=pretaxannualwage*7/365.2425 if em08d==2|em11d==2
replace wage=grosswage

* Deflating wages to 2015
gen index=.
replace index=75.0 if yearofinterview==2004
replace index=77.1 if yearofinterview==2005
replace index=79.8 if yearofinterview==2006
replace index=81.7 if yearofinterview==2007
replace index=85.2 if yearofinterview==2008
replace index=86.7 if yearofinterview==2009
replace index=89.25006 if yearofinterview==2010
replace index=92.19875 if yearofinterview==2011
replace index=93.82401 if yearofinterview==2012
replace index=96.12259 if yearofinterview==2013
replace index=98.51405 if yearofinterview==2014
replace index=100 if yearofinterview==2015
replace index=101.277 if yearofinterview==2016
replace index=103.2505 if yearofinterview==2017
replace wage=wage*100/index

* Count numbers of nonmissing observations
gen Nemp=!missing(employment)
gen Ninc=!missing(wage)

preserve
collapse (mean) employment avg_income=wage (rawsum) Nemp Ninc [aw=indweight], by(female yearssincearrive migranttype)
drop if missing(yearssincearrive)
save "AU-BNLA", replace
restore
preserve
collapse (mean) employment avg_income=wage (rawsum) Nemp Ninc [aw=indweight], by(yearssincearrive migranttype)
drop if missing(yearssincearrive)
append using "AU-BNLA"
order yearssincearrive migranttype female employment Nemp avg_income Ninc
sort female yearssincearrive
save "AU-BNLA", replace
restore


********************************************************************************
*** Calculate sample descriptives

* # Observations
count
* # Unique individuals
egen persTag=tag(xwaveid)
tab persTag

* Gender
sum female [aw=indweight], detail
* Age
sum age [aw=indweight], detail

* Country of origin
preserve
	keep if migranttype==1
	keep if time_since_arrival>=1&time_since_arrival<=9
	capture gen dummy=1
	collapse (sum) numrefugees=dummy [iw=indweight], by(countryoforigin)
	egen totalrefugees=sum(numrefugees)
	gen fracrefugees=numrefugees/totalrefugees
	gsort -fracrefugees
	gen thecounter=_n
	list countryoforigin fracrefugees if thecounter<=5
restore

* LM outcomes
sum employment [aw=indweight], detail
sum wage [aw=indweight], detail

********************************************************************************
*** Clean up

capture noisily erase "temp.dta"
clear all
